<?php
session_start();

// Carrega configurações de banco de dados
require_once __DIR__ . '/config/database.php';

// Função para limpar dados de entrada
function cleanInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}

// Processamento do formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = cleanInput($_POST['email'] ?? '');
    $senha = cleanInput($_POST['senha'] ?? '');

    try {
        // Conexão usando configuração centralizada
        $pdo = getDBConnection();

        // Buscar usuário
        $stmt = $pdo->prepare("SELECT * FROM usuarios WHERE email = ? AND ativo = 1");
        $stmt->execute([$email]);
        $usuario = $stmt->fetch();

        if ($usuario && password_verify($senha, $usuario['senha'])) {
            // Login bem-sucedido
            $_SESSION['logado'] = true;
            $_SESSION['user_id'] = $usuario['id'];
            $_SESSION['user_email'] = $usuario['email'];
            $_SESSION['user_nome'] = $usuario['nome'];
            $_SESSION['user_nivel'] = $usuario['nivel'];
            
            // Atualizar último login
            $stmt = $pdo->prepare("UPDATE usuarios SET ultimo_login = NOW() WHERE id = ?");
            $stmt->execute([$usuario['id']]);
            
            header('Location: dashboard.php');
            exit;
        } else {
            $_SESSION['erro'] = 'Email ou senha inválidos';
        }
    } catch (PDOException $e) {
        $_SESSION['erro'] = 'Erro de conexão com o banco de dados';
        error_log("Erro de login: " . $e->getMessage());
    }
}

// Buscar configurações do sistema
try {
    $pdo_config = getDBConnection();
    $stmt = $pdo_config->query("SELECT * FROM configuracoes WHERE id = 1");
    $config = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Se não encontrar configuração, usar valores padrão
    if (!$config) {
        $config = [
            'titulo_site' => 'SIMAE',
            'logo_url' => '',
            'cor_primaria' => '#16a34a',
            'cor_secundaria' => '#15803d'
        ];
    }
} catch (Exception $e) {
    $config = [
        'titulo_site' => 'SIMAE',
        'logo_url' => '',
        'cor_primaria' => '#16a34a',
        'cor_secundaria' => '#15803d'
    ];
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($config['site_title'] ?? 'SIMAE'); ?> - Login Administrativo</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#f0fdf4',
                            100: '#dcfce7',
                            200: '#bbf7d0',
                            300: '#86efac',
                            400: '#4ade80',
                            500: '#22c55e',
                            600: '#16a34a',
                            700: '#15803d',
                            800: '#166534',
                            900: '#14532d',
                        }
                    }
                }
            }
        }
    </script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .gradient-bg { background: linear-gradient(135deg, #f0fdf4 0%, #dcfce7 50%, #bbf7d0 100%); }
    </style>
</head>
<body class="gradient-bg min-h-screen flex items-center justify-center">
    <div class="w-full max-w-md p-8">
        <!-- Logo e Título -->
        <div class="text-center mb-8">
            <div class="inline-flex items-center justify-center w-20 h-20 bg-gradient-to-r from-primary-600 to-primary-700 rounded-2xl mb-4">
                <?php if (!empty($config['logo_url'])): ?>
                    <img src="<?php echo htmlspecialchars($config['logo_url']); ?>" alt="Logo" class="w-16 h-16 object-contain">
                <?php else: ?>
                    <i class="fas fa-shield-alt text-white text-3xl"></i>
                <?php endif; ?>
            </div>
            <h1 class="text-3xl font-bold text-gray-800"><?php echo htmlspecialchars($config['titulo_site'] ?? 'SIMAE'); ?></h1>
            <p class="text-gray-600">Painel Administrativo</p>
        </div>

        <!-- Formulário de Login -->
        <div class="bg-white rounded-2xl shadow-xl p-8 mb-6">
            <form action="login.php" method="POST" id="loginForm">
                <!-- Campo Email -->
                <div class="mb-6">
                    <label class="block text-gray-700 text-sm font-semibold mb-2" for="email">
                        <i class="fas fa-envelope text-primary-500 mr-2"></i>E-mail
                    </label>
                    <input type="email" id="email" name="email" 
                           class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-primary-500 focus:ring-2 focus:ring-primary-200 outline-none transition-all"
                           placeholder="Digite seu email" required>
                </div>

                <!-- Campo Senha -->
                <div class="mb-6">
                    <label class="block text-gray-700 text-sm font-semibold mb-2" for="senha">
                        <i class="fas fa-lock text-primary-500 mr-2"></i>Senha
                    </label>
                    <div class="relative">
                        <input type="password" id="senha" name="senha" 
                               class="w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-primary-500 focus:ring-2 focus:ring-primary-200 outline-none transition-all"
                               placeholder="Digite sua senha" required>
                        <button type="button" onclick="togglePassword()" 
                                class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-primary-500">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>

                <?php if (isset($_SESSION['erro'])): ?>
                <div class="mb-6 p-4 bg-red-50 text-red-700 rounded-lg">
                    <i class="fas fa-exclamation-circle mr-2"></i>
                    <?php echo $_SESSION['erro']; unset($_SESSION['erro']); ?>
                </div>
                <?php endif; ?>

                <!-- Botão Entrar -->
                <button type="submit" 
                        class="w-full bg-gradient-to-r from-primary-600 to-primary-700 hover:from-primary-700 hover:to-primary-800 text-white font-semibold py-3 px-6 rounded-xl transition-all duration-300 transform hover:scale-[1.02] hover:shadow-lg">
                    <i class="fas fa-sign-in-alt mr-2"></i>Entrar no Sistema
                </button>
            </form>
        </div>

        <!-- Links e Copyright -->
        <div class="text-center">
            <a href="../" class="inline-flex items-center text-primary-600 hover:text-primary-700 font-semibold mb-4">
                <i class="fas fa-arrow-left mr-2"></i>Voltar ao início
            </a>
            <p class="text-gray-500 text-sm">
                © 2025 SIMAE - Sistema de Administração
            </p>
        </div>
    </div>

    <script>
        // Previne múltiplos submits
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            const submitButton = this.querySelector('button[type="submit"]');
            submitButton.disabled = true;
            submitButton.innerHTML = '<i class="fas fa-circle-notch fa-spin mr-2"></i>Entrando...';
        });

        function togglePassword() {
            const senhaInput = document.getElementById('senha');
            const type = senhaInput.getAttribute('type') === 'password' ? 'text' : 'password';
            senhaInput.setAttribute('type', type);
            
            const icon = event.currentTarget.querySelector('i');
            icon.classList.toggle('fa-eye');
            icon.classList.toggle('fa-eye-slash');
        }
    </script>
</body>
</html>