<?php
// Carrega configurações e inicia sessão
require_once __DIR__ . '/../includes/init.php';

// Garantir tabelas necessárias
$pdo->exec("CREATE TABLE IF NOT EXISTS escolas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uniq_nome (nome)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

$pdo->exec("CREATE TABLE IF NOT EXISTS escola_series (
    id INT AUTO_INCREMENT PRIMARY KEY,
    escola_id INT NOT NULL,
    nome_serie VARCHAR(50) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uniq_escola_serie (escola_id, nome_serie),
    KEY idx_escola (escola_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

$pdo->exec("CREATE TABLE IF NOT EXISTS turmas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    escola_id INT NOT NULL,
    serie_id INT NOT NULL,
    nome_turma VARCHAR(50) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uniq_turma (escola_id, serie_id, nome_turma),
    KEY idx_escola (escola_id),
    KEY idx_serie (serie_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

// Helpers
function get_escolas($pdo) {
    return $pdo->query('SELECT * FROM escolas ORDER BY nome')->fetchAll();
}
function get_series_by_escola($pdo, $escola_id) {
    $st = $pdo->prepare('SELECT * FROM escola_series WHERE escola_id = ? ORDER BY nome_serie');
    $st->execute(array($escola_id));
    return $st->fetchAll();
}
function get_turmas($pdo, $escola_id, $serie_id) {
    $st = $pdo->prepare('SELECT * FROM turmas WHERE escola_id = ? AND serie_id = ? ORDER BY nome_turma');
    $st->execute(array($escola_id, $serie_id));
    return $st->fetchAll();
}

// Mensagens
$msg_ok = '';
$msg_err = '';

// Ações: adicionar/remover
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'add_escola') {
        $nome = trim($_POST['nome'] ?? '');
        if ($nome !== '') {
            try {
                $st = $pdo->prepare('INSERT INTO escolas (nome) VALUES (?)');
                $st->execute(array($nome));
                $msg_ok = 'Escola cadastrada com sucesso.';
            } catch (PDOException $e) {
                $msg_err = 'Não foi possível cadastrar a escola (nome duplicado?).';
            }
        } else {
            $msg_err = 'Informe o nome da escola.';
        }
    }

    if ($action === 'add_serie') {
        $escola_id = (int)($_POST['escola_id'] ?? 0);
        $nome_serie = trim($_POST['nome_serie'] ?? '');
        if ($escola_id && $nome_serie !== '') {
            try {
                $st = $pdo->prepare('INSERT INTO escola_series (escola_id, nome_serie) VALUES (?, ?)');
                $st->execute(array($escola_id, $nome_serie));
                $msg_ok = 'Série cadastrada para a escola.';
            } catch (PDOException $e) {
                $msg_err = 'Não foi possível cadastrar a série (duplicada?).';
            }
        } else {
            $msg_err = 'Selecione a escola e informe a série.';
        }
    }

    if ($action === 'add_turma') {
        $escola_id = (int)($_POST['escola_id'] ?? 0);
        $serie_id = (int)($_POST['serie_id'] ?? 0);
        $nome_turma = trim($_POST['nome_turma'] ?? '');
        if ($escola_id && $serie_id && $nome_turma !== '') {
            try {
                $st = $pdo->prepare('INSERT INTO turmas (escola_id, serie_id, nome_turma) VALUES (?, ?, ?)');
                $st->execute(array($escola_id, $serie_id, $nome_turma));
                $msg_ok = 'Turma cadastrada com sucesso.';
            } catch (PDOException $e) {
                $msg_err = 'Não foi possível cadastrar a turma (duplicada?).';
            }
        } else {
            $msg_err = 'Selecione a escola, a série e informe o nome da turma.';
        }
    }
}

// Exclusões via GET (com redirect para limpar query)
if (($_GET['action'] ?? '') === 'delete_serie' && !empty($_GET['id'])) {
    $id = (int)$_GET['id'];
    // Apagar turmas da série antes
    $st = $pdo->prepare('DELETE FROM turmas WHERE serie_id = ?');
    $st->execute(array($id));
    // Apagar série
    $pdo->prepare('DELETE FROM escola_series WHERE id = ?')->execute(array($id));
    header('Location: ?page=escolas_turmas');
    exit;
}
if (($_GET['action'] ?? '') === 'delete_turma' && !empty($_GET['id'])) {
    $id = (int)$_GET['id'];
    $pdo->prepare('DELETE FROM turmas WHERE id = ?')->execute(array($id));
    header('Location: ?page=escolas_turmas');
    exit;
}
if (($_GET['action'] ?? '') === 'delete_escola' && !empty($_GET['id'])) {
    $id = (int)$_GET['id'];
    // Cascata manual: turmas -> series -> escola
    $series = get_series_by_escola($pdo, $id);
    foreach ($series as $s) {
        $pdo->prepare('DELETE FROM turmas WHERE serie_id = ?')->execute(array($s['id']));
    }
    $pdo->prepare('DELETE FROM escola_series WHERE escola_id = ?')->execute(array($id));
    $pdo->prepare('DELETE FROM escolas WHERE id = ?')->execute(array($id));
    header('Location: ?page=escolas_turmas');
    exit;
}

// Filtros da UI
$sel_escola_series = isset($_GET['escola_series']) ? (int)$_GET['escola_series'] : 0; // para seção "Séries"
$sel_escola_turmas = isset($_GET['escola_turmas']) ? (int)$_GET['escola_turmas'] : 0; // para seção "Turmas"
$sel_serie_turmas  = isset($_GET['serie_turmas']) ? (int)$_GET['serie_turmas'] : 0;

$lista_escolas = get_escolas($pdo);
$lista_series_sec = $sel_escola_series ? get_series_by_escola($pdo, $sel_escola_series) : array();
$lista_series_turmas = $sel_escola_turmas ? get_series_by_escola($pdo, $sel_escola_turmas) : array();
$lista_turmas = ($sel_escola_turmas && $sel_serie_turmas) ? get_turmas($pdo, $sel_escola_turmas, $sel_serie_turmas) : array();

$series_sugestoes = array('1º ANO','2º ANO','3º ANO','4º ANO','5º ANO');
?>

<div class="max-w-full">
    <div class="bg-white shadow rounded-lg mb-6">
        <div class="px-6 py-4 border-b border-gray-200">
            <h1 class="text-2xl font-bold text-gray-900">
                <i class="fas fa-school mr-3 text-primary-600"></i>
                Escolas, Séries e Turmas
            </h1>
            <p class="text-gray-600 mt-1">Cadastre escolas, defina séries por escola e crie turmas por série.</p>
        </div>
        <div class="p-6">
            <?php if ($msg_ok): ?><div class="mb-4 p-3 bg-green-100 text-green-700 rounded"><?php echo $msg_ok; ?></div><?php endif; ?>
            <?php if ($msg_err): ?><div class="mb-4 p-3 bg-red-100 text-red-700 rounded"><?php echo $msg_err; ?></div><?php endif; ?>

            <!-- Seção 1: Cadastrar Escola -->
            <div class="mb-8">
                <h2 class="text-lg font-semibold mb-3"><i class="fas fa-plus-circle mr-2 text-blue-600"></i>Nova Escola</h2>
                <form method="POST" class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <input type="hidden" name="action" value="add_escola">
                    <div class="md:col-span-2">
                        <label class="block text-sm font-medium text-gray-700">Nome da Escola</label>
                        <input type="text" name="nome" class="mt-1 block w-full border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500" placeholder="Ex.: EMEF Maria Silva" required>
                    </div>
                    <div class="flex items-end">
                        <button class="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700" type="submit">Cadastrar</button>
                    </div>
                </form>

                <?php if (!empty($lista_escolas)): ?>
                <div class="mt-4 overflow-x-auto">
                    <table class="min-w-full border border-gray-200 text-sm">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="border p-2 text-left">Escola</th>
                                <th class="border p-2 text-center">Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($lista_escolas as $esc): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="border p-2"><?php echo htmlspecialchars($esc['nome']); ?></td>
                                <td class="border p-2 text-center">
                                    <a class="text-red-600 hover:text-red-800" onclick="return confirm('Excluir esta escola e todos os registros relacionados?');" href="?page=escolas_turmas&action=delete_escola&id=<?php echo $esc['id']; ?>">Excluir</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>

            <hr class="my-6">

            <!-- Seção 2: Séries por Escola -->
            <div class="mb-8">
                <h2 class="text-lg font-semibold mb-3"><i class="fas fa-layer-group mr-2 text-indigo-600"></i>Séries por Escola</h2>
                <form method="GET" class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-3">
                    <input type="hidden" name="page" value="escolas_turmas">
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Escola</label>
                        <select name="escola_series" class="mt-1 block w-full border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500" onchange="this.form.submit()">
                            <option value="">Selecione...</option>
                            <?php foreach ($lista_escolas as $esc): ?>
                            <option value="<?php echo $esc['id']; ?>" <?php echo $sel_escola_series == $esc['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($esc['nome']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </form>

                <?php if ($sel_escola_series): ?>
                <form method="POST" class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <input type="hidden" name="action" value="add_serie">
                    <input type="hidden" name="escola_id" value="<?php echo $sel_escola_series; ?>">
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Série</label>
                        <input list="sug_series" name="nome_serie" class="mt-1 block w-full border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500" placeholder="Ex.: 1º ANO" required>
                        <datalist id="sug_series">
                            <?php foreach ($series_sugestoes as $sug): ?>
                            <option value="<?php echo $sug; ?>"></option>
                            <?php endforeach; ?>
                        </datalist>
                    </div>
                    <div class="flex items-end">
                        <button class="bg-indigo-600 text-white px-4 py-2 rounded hover:bg-indigo-700" type="submit">Adicionar Série</button>
                    </div>
                </form>

                <div class="mt-4 overflow-x-auto">
                    <table class="min-w-full border border-gray-200 text-sm">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="border p-2 text-left">Série</th>
                                <th class="border p-2 text-center">Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($lista_series_sec as $s): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="border p-2"><?php echo htmlspecialchars($s['nome_serie']); ?></td>
                                <td class="border p-2 text-center">
                                    <a class="text-red-600 hover:text-red-800" onclick="return confirm('Excluir esta série e suas turmas?');" href="?page=escolas_turmas&action=delete_serie&id=<?php echo $s['id']; ?>">Excluir</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>

            <hr class="my-6">

            <!-- Seção 3: Turmas por Série da Escola -->
            <div>
                <h2 class="text-lg font-semibold mb-3"><i class="fas fa-people-group mr-2 text-emerald-600"></i>Turmas por Série</h2>
                <form method="GET" class="grid grid-cols-1 md:grid-cols-5 gap-4 mb-3">
                    <input type="hidden" name="page" value="escolas_turmas">
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Escola</label>
                        <select name="escola_turmas" class="mt-1 block w-full border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500" onchange="this.form.submit()">
                            <option value="">Selecione...</option>
                            <?php foreach ($lista_escolas as $esc): ?>
                            <option value="<?php echo $esc['id']; ?>" <?php echo $sel_escola_turmas == $esc['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($esc['nome']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Série</label>
                        <select name="serie_turmas" class="mt-1 block w-full border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500" onchange="this.form.submit()">
                            <option value="">Selecione...</option>
                            <?php foreach ($lista_series_turmas as $s): ?>
                            <option value="<?php echo $s['id']; ?>" <?php echo $sel_serie_turmas == $s['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($s['nome_serie']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </form>

                <?php if ($sel_escola_turmas && $sel_serie_turmas): ?>
                <form method="POST" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <input type="hidden" name="action" value="add_turma">
                    <input type="hidden" name="escola_id" value="<?php echo $sel_escola_turmas; ?>">
                    <input type="hidden" name="serie_id" value="<?php echo $sel_serie_turmas; ?>">
                    <div class="md:col-span-2">
                        <label class="block text-sm font-medium text-gray-700">Nome da Turma</label>
                        <input type="text" name="nome_turma" class="mt-1 block w-full border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500" placeholder="Ex.: A, B, C" required>
                    </div>
                    <div class="flex items-end">
                        <button class="bg-emerald-600 text-white px-4 py-2 rounded hover:bg-emerald-700" type="submit">Adicionar Turma</button>
                    </div>
                </form>

                <div class="mt-4 overflow-x-auto">
                    <table class="min-w-full border border-gray-200 text-sm">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="border p-2 text-left">Turma</th>
                                <th class="border p-2 text-center">Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($lista_turmas as $t): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="border p-2"><?php echo htmlspecialchars($t['nome_turma']); ?></td>
                                <td class="border p-2 text-center">
                                    <a class="text-red-600 hover:text-red-800" onclick="return confirm('Excluir esta turma?');" href="?page=escolas_turmas&action=delete_turma&id=<?php echo $t['id']; ?>">Excluir</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<style>
@media print {
    .no-print { display: none; }
}
</style>
