<?php
// Proteção da página
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// O $pdo já está disponível do dashboard.php

// Buscar todas as escolas
$st = $pdo->query('SELECT id, nome FROM escolas ORDER BY nome');
$escolas = $st->fetchAll(PDO::FETCH_ASSOC);

// Tipo de relatório selecionado
$tipo_relatorio = $_GET['tipo'] ?? '';
$escola_id = $_GET['escola'] ?? '';
$serie_id = $_GET['serie'] ?? '';
$turma_id = $_GET['turma'] ?? '';
$bimestre = $_GET['bimestre'] ?? '';
$ano_letivo = $_GET['ano_letivo'] ?? date('Y');
$modo_visualizacao = $_GET['modo'] ?? 'individual'; // 'individual', 'todos', 'geral'

// Buscar séries da escola selecionada
$series_db = [];
if ($escola_id) {
    $st = $pdo->prepare("SELECT id, nome_serie FROM escola_series WHERE escola_id = ? ORDER BY nome_serie");
    $st->execute([$escola_id]);
    $series_db = $st->fetchAll(PDO::FETCH_ASSOC);
}

// Buscar turmas
$turmas_db = [];
if ($escola_id && $serie_id) {
    $st = $pdo->prepare('SELECT id, nome_turma FROM turmas WHERE escola_id = ? AND serie_id = ? ORDER BY nome_turma');
    $st->execute([$escola_id, $serie_id]);
    $turmas_db = $st->fetchAll(PDO::FETCH_ASSOC);
}

// Buscar nomes para exibição
$escola_nome = '';
$serie_nome = '';
$turma_nome = '';

if ($escola_id) {
    $st = $pdo->prepare('SELECT nome FROM escolas WHERE id = ?');
    $st->execute([$escola_id]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    if ($row) $escola_nome = $row['nome'];
}

if ($serie_id) {
    $st = $pdo->prepare('SELECT nome_serie FROM escola_series WHERE id = ?');
    $st->execute([$serie_id]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    if ($row) $serie_nome = $row['nome_serie'];
}

if ($turma_id) {
    $st = $pdo->prepare('SELECT nome_turma FROM turmas WHERE id = ?');
    $st->execute([$turma_id]);
    $row = $st->fetch(PDO::FETCH_ASSOC);
    if ($row) $turma_nome = $row['nome_turma'];
}

// Buscar dados para modo "todos" (todas as turmas de uma escola)
$dados_relatorios = [];
if ($modo_visualizacao === 'todos' && $tipo_relatorio && $escola_id) {
    $tabelas = [
        'leitura_iniciais' => 'condensado_leitura_turma',
        'matematica_iniciais' => 'condensado_matematica_turma',
        'supervisor' => 'condensado_supervisor_maternal',
        'leitura_finais' => 'condensado_leitura_finais',
        'redacao_finais' => 'condensado_redacao_finais',
        'matematica_finais' => 'condensado_matematica_finais'
    ];
    
    $tabela = $tabelas[$tipo_relatorio] ?? '';
    
    if ($tabela) {
        // Supervisor não tem serie_id, então JOIN apenas com turmas
        if ($tipo_relatorio === 'supervisor') {
            $query = "SELECT c.*, '' as nome_serie, t.nome_turma 
                      FROM {$tabela} c
                      JOIN turmas t ON c.turma_id = t.id
                      WHERE c.escola_id = ? AND c.ano_letivo = ?";
        } else {
            $query = "SELECT c.*, es.nome_serie, t.nome_turma 
                      FROM {$tabela} c
                      JOIN escola_series es ON c.serie_id = es.id
                      JOIN turmas t ON c.turma_id = t.id
                      WHERE c.escola_id = ? AND c.ano_letivo = ?";
        }
        
        // Definir o ORDER BY correto
        $order_by = ($tipo_relatorio === 'supervisor') ? 't.nome_turma' : 'es.nome_serie, t.nome_turma';
        
        if ($bimestre) {
            $query .= " AND c.bimestre = ?";
            $st = $pdo->prepare($query . " ORDER BY {$order_by}");
            $st->execute([$escola_id, $ano_letivo, $bimestre]);
        } else {
            $st = $pdo->prepare($query . " ORDER BY {$order_by}");
            $st->execute([$escola_id, $ano_letivo]);
        }
        
        $dados_relatorios = $st->fetchAll(PDO::FETCH_ASSOC);
    }
}

// Buscar dados para relatório geral (resumo estatístico)
$dados_geral = [];
if ($modo_visualizacao === 'geral' && $tipo_relatorio && $escola_id) {
    $tabelas = [
        'leitura_iniciais' => 'condensado_leitura_turma',
        'matematica_iniciais' => 'condensado_matematica_turma',
        'supervisor' => 'condensado_supervisor_maternal',
        'leitura_finais' => 'condensado_leitura_finais',
        'redacao_finais' => 'condensado_redacao_finais',
        'matematica_finais' => 'condensado_matematica_finais'
    ];
    
    // Definir o nome correto da coluna de necessidades especiais por tipo
    $coluna_necessidades = [
        'leitura_iniciais' => 'necessidades_especiais',
        'matematica_iniciais' => 'necessidades_especiais',
        'supervisor' => 'alunos_com_laudo',
        'leitura_finais' => 'alunos_necessidades_especiais',
        'redacao_finais' => 'alunos_necessidades_especiais',
        'matematica_finais' => 'alunos_necessidades_especiais'
    ];
    
    $tabela = $tabelas[$tipo_relatorio] ?? '';
    $coluna_nec = $coluna_necessidades[$tipo_relatorio] ?? 'alunos_necessidades_especiais';
    
    if ($tabela) {
        // Supervisor não tem serie_id, então busca direto por turma
        if ($tipo_relatorio === 'supervisor') {
            $query = "SELECT t.nome_turma as nome_serie, 
                             COUNT(DISTINCT c.turma_id) as total_turmas,
                             SUM(c.total_alunos) as total_alunos,
                             SUM(c.{$coluna_nec}) as total_necessidades
                      FROM {$tabela} c
                      JOIN turmas t ON c.turma_id = t.id
                      WHERE c.escola_id = ? AND c.ano_letivo = ?";
        } else {
            // Buscar totais por série (para os outros tipos)
            $query = "SELECT es.nome_serie, 
                             COUNT(DISTINCT c.turma_id) as total_turmas,
                             SUM(c.total_alunos) as total_alunos,
                             SUM(c.{$coluna_nec}) as total_necessidades
                      FROM {$tabela} c
                      JOIN escola_series es ON c.serie_id = es.id
                      WHERE c.escola_id = ? AND c.ano_letivo = ?";
        }
        
        // Definir o GROUP BY correto
        $group_by_field = ($tipo_relatorio === 'supervisor') ? 't.nome_turma' : 'es.nome_serie';
        
        if ($bimestre) {
            $query .= " AND c.bimestre = ?";
            $st = $pdo->prepare($query . " GROUP BY {$group_by_field} ORDER BY {$group_by_field}");
            $st->execute([$escola_id, $ano_letivo, $bimestre]);
        } else {
            $st = $pdo->prepare($query . " GROUP BY {$group_by_field} ORDER BY {$group_by_field}");
            $st->execute([$escola_id, $ano_letivo]);
        }
        
        $dados_geral = $st->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>

<div class="max-w-full">
    <div class="bg-white shadow rounded-lg mb-6">
        <div class="bg-gradient-to-r from-teal-600 to-cyan-800 px-6 py-4 border-b border-gray-200">
            <div class="flex justify-between items-center">
                <div>
                    <h1 class="text-2xl font-bold text-white">
                    <i class="fas fa-chart-bar mr-3"></i>
                    SISTEMA DE RELATÓRIOS
                    </h1>
                    <p class="text-teal-100 mt-1">Relatórios individuais, por escola e estatísticas gerais</p>
                </div>
            </div>
        </div>

        <div class="p-6">
            <!-- Seleção do Tipo de Relatório (PRIMEIRO) -->
            <div class="bg-gradient-to-r from-teal-50 to-cyan-50 border-2 border-teal-200 rounded-lg p-6 mb-6">
                <h3 class="text-xl font-bold text-teal-800 mb-2">
                    <i class="fas fa-clipboard-list mr-2"></i>1. Selecione o Tipo de Condensado
                </h3>
                <p class="text-sm text-gray-600 mb-4">Escolha qual condensado deseja gerar relatório</p>
                <form method="GET" id="formTipoRelatorio">
                    <input type="hidden" name="page" value="relatorios">
                    <input type="hidden" name="modo" value="<?php echo htmlspecialchars($modo_visualizacao); ?>">
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                        <label class="cursor-pointer bg-white border-2 <?php echo $tipo_relatorio === 'leitura_iniciais' ? 'border-blue-500 bg-blue-50' : 'border-gray-300'; ?> rounded-lg p-4 hover:border-blue-400 transition-all">
                            <input type="radio" name="tipo" value="leitura_iniciais" <?php echo $tipo_relatorio === 'leitura_iniciais' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-2">
                            <span class="font-semibold text-blue-700">📘 Leitura e Escrita</span>
                            <span class="text-xs text-gray-600 block mt-1">Anos Iniciais (1º-5º)</span>
                        </label>
                        <label class="cursor-pointer bg-white border-2 <?php echo $tipo_relatorio === 'matematica_iniciais' ? 'border-green-500 bg-green-50' : 'border-gray-300'; ?> rounded-lg p-4 hover:border-green-400 transition-all">
                            <input type="radio" name="tipo" value="matematica_iniciais" <?php echo $tipo_relatorio === 'matematica_iniciais' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-2">
                            <span class="font-semibold text-green-700">📗 Matemática</span>
                            <span class="text-xs text-gray-600 block mt-1">Anos Iniciais (1º-5º)</span>
                        </label>
                        <label class="cursor-pointer bg-white border-2 <?php echo $tipo_relatorio === 'supervisor' ? 'border-purple-500 bg-purple-50' : 'border-gray-300'; ?> rounded-lg p-4 hover:border-purple-400 transition-all">
                            <input type="radio" name="tipo" value="supervisor" <?php echo $tipo_relatorio === 'supervisor' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-2">
                            <span class="font-semibold text-purple-700">🟣 Supervisor</span>
                            <span class="text-xs text-gray-600 block mt-1">Educação Infantil</span>
                        </label>
                        <label class="cursor-pointer bg-white border-2 <?php echo $tipo_relatorio === 'leitura_finais' ? 'border-blue-500 bg-blue-50' : 'border-gray-300'; ?> rounded-lg p-4 hover:border-blue-400 transition-all">
                            <input type="radio" name="tipo" value="leitura_finais" <?php echo $tipo_relatorio === 'leitura_finais' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-2">
                            <span class="font-semibold text-blue-700">📘 Leitura e Escrita</span>
                            <span class="text-xs text-gray-600 block mt-1">Anos Finais (6º-9º)</span>
                        </label>
                        <label class="cursor-pointer bg-white border-2 <?php echo $tipo_relatorio === 'redacao_finais' ? 'border-indigo-500 bg-indigo-50' : 'border-gray-300'; ?> rounded-lg p-4 hover:border-indigo-400 transition-all">
                            <input type="radio" name="tipo" value="redacao_finais" <?php echo $tipo_relatorio === 'redacao_finais' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-2">
                            <span class="font-semibold text-indigo-700">📝 Redação</span>
                            <span class="text-xs text-gray-600 block mt-1">Anos Finais (6º-9º)</span>
                        </label>
                        <label class="cursor-pointer bg-white border-2 <?php echo $tipo_relatorio === 'matematica_finais' ? 'border-green-500 bg-green-50' : 'border-gray-300'; ?> rounded-lg p-4 hover:border-green-400 transition-all">
                            <input type="radio" name="tipo" value="matematica_finais" <?php echo $tipo_relatorio === 'matematica_finais' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-2">
                            <span class="font-semibold text-green-700">📗 Matemática</span>
                            <span class="text-xs text-gray-600 block mt-1">Anos Finais (6º-9º)</span>
                        </label>
                    </div>
                </form>
            </div>

            <?php if ($tipo_relatorio): ?>
            <!-- Modo de Visualização (SEGUNDO) -->
            <div class="bg-gradient-to-r from-indigo-50 to-purple-50 border-2 border-indigo-200 rounded-lg p-6 mb-6">
                <h3 class="text-xl font-bold text-indigo-800 mb-2">
                    <i class="fas fa-eye mr-2"></i>2. Escolha o Modo de Visualização
                </h3>
                <p class="text-sm text-gray-600 mb-4">Como deseja visualizar os dados do relatório</p>
                <form method="GET" id="formModoVisualizacao">
                    <input type="hidden" name="page" value="relatorios">
                    <input type="hidden" name="tipo" value="<?php echo htmlspecialchars($tipo_relatorio); ?>">
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <label class="cursor-pointer bg-white border-2 <?php echo $modo_visualizacao === 'individual' ? 'border-indigo-500 bg-indigo-50 shadow-lg' : 'border-gray-300'; ?> rounded-lg p-5 hover:border-indigo-400 hover:shadow-md transition-all">
                            <input type="radio" name="modo" value="individual" <?php echo $modo_visualizacao === 'individual' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-3">
                            <div>
                                <div class="font-bold text-lg text-indigo-700">
                                    <i class="fas fa-user-graduate mr-2"></i>Relatório Individual
                                </div>
                                <div class="text-sm text-gray-600 mt-2">Gerar relatório de <strong>uma turma específica</strong></div>
                            </div>
                        </label>
                        <label class="cursor-pointer bg-white border-2 <?php echo $modo_visualizacao === 'todos' ? 'border-purple-500 bg-purple-50 shadow-lg' : 'border-gray-300'; ?> rounded-lg p-5 hover:border-purple-400 hover:shadow-md transition-all">
                            <input type="radio" name="modo" value="todos" <?php echo $modo_visualizacao === 'todos' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-3">
                            <div>
                                <div class="font-bold text-lg text-purple-700">
                                    <i class="fas fa-list-ul mr-2"></i>Lista de Turmas
                                </div>
                                <div class="text-sm text-gray-600 mt-2">Listar <strong>todas as turmas</strong> da escola com ações</div>
                            </div>
                        </label>
                        <label class="cursor-pointer bg-white border-2 <?php echo $modo_visualizacao === 'geral' ? 'border-pink-500 bg-pink-50 shadow-lg' : 'border-gray-300'; ?> rounded-lg p-5 hover:border-pink-400 hover:shadow-md transition-all">
                            <input type="radio" name="modo" value="geral" <?php echo $modo_visualizacao === 'geral' ? 'checked' : ''; ?> onchange="this.form.submit()" class="mr-3">
                            <div>
                                <div class="font-bold text-lg text-pink-700">
                                    <i class="fas fa-chart-pie mr-2"></i>Estatísticas Gerais
                                </div>
                                <div class="text-sm text-gray-600 mt-2">Ver <strong>resumo estatístico</strong> por série/período</div>
                            </div>
                        </label>
                    </div>
                </form>
            </div>
            <?php endif; ?>

            <?php if ($tipo_relatorio): ?>
            <!-- Filtros de Escola, Série, Turma, Bimestre -->
            <div class="bg-gradient-to-r from-gray-50 to-slate-50 border-2 border-gray-300 rounded-lg p-6 mb-6">
                <h3 class="text-xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-sliders-h mr-2"></i>3. Defina os Filtros
                </h3>
                <p class="text-sm text-gray-600 mb-4">Configure os filtros para gerar o relatório</p>
                <form method="GET" class="grid grid-cols-1 md:grid-cols-<?php echo $modo_visualizacao === 'individual' ? '6' : '4'; ?> gap-4">
                    <input type="hidden" name="page" value="relatorios">
                    <input type="hidden" name="tipo" value="<?php echo htmlspecialchars($tipo_relatorio); ?>">
                    <input type="hidden" name="modo" value="<?php echo htmlspecialchars($modo_visualizacao); ?>">
                    
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">
                            <i class="fas fa-school mr-1 text-teal-600"></i>Escola
                        </label>
                        <select name="escola" class="block w-full border-2 border-gray-400 rounded-md p-3 text-base font-semibold focus:ring-2 focus:ring-teal-500 focus:border-teal-500 bg-white" onchange="this.form.submit()">
                            <option value="">Selecione...</option>
                            <?php foreach ($escolas as $esc): ?>
                            <option value="<?php echo $esc['id']; ?>" <?php echo $escola_id == $esc['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($esc['nome']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">
                            <i class="fas fa-layer-group mr-1 text-teal-600"></i>Série
                        </label>
                        <select name="serie" class="block w-full border-2 border-gray-400 rounded-md p-3 text-base font-semibold focus:ring-2 focus:ring-teal-500 focus:border-teal-500 bg-white" onchange="this.form.submit()">
                            <option value="">Selecione...</option>
                            <?php foreach ($series_db as $s): ?>
                            <option value="<?php echo $s['id']; ?>" <?php echo $serie_id == $s['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($s['nome_serie']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <?php if ($modo_visualizacao === 'individual'): ?>
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">
                            <i class="fas fa-users mr-1 text-teal-600"></i>Turma
                        </label>
                        <select name="turma" class="block w-full border-2 border-gray-400 rounded-md p-3 text-base font-semibold focus:ring-2 focus:ring-teal-500 focus:border-teal-500 bg-white" onchange="this.form.submit()">
                            <option value="">Selecione...</option>
                            <?php foreach ($turmas_db as $t): ?>
                            <option value="<?php echo $t['id']; ?>" <?php echo $turma_id == $t['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($t['nome_turma']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php endif; ?>
                    
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">
                            <i class="fas fa-calendar-check mr-1 text-teal-600"></i>Bimestre
                        </label>
                        <select name="bimestre" class="block w-full border-2 border-gray-400 rounded-md p-3 text-base font-semibold focus:ring-2 focus:ring-teal-500 focus:border-teal-500 bg-white">
                            <option value="">Todos os Bimestres</option>
                            <option value="1" <?php echo $bimestre == '1' ? 'selected' : ''; ?>>1º Bimestre</option>
                            <option value="2" <?php echo $bimestre == '2' ? 'selected' : ''; ?>>2º Bimestre</option>
                            <option value="3" <?php echo $bimestre == '3' ? 'selected' : ''; ?>>3º Bimestre</option>
                            <option value="4" <?php echo $bimestre == '4' ? 'selected' : ''; ?>>4º Bimestre</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">
                            <i class="fas fa-calendar-alt mr-1 text-teal-600"></i>Ano Letivo
                        </label>
                        <select name="ano_letivo" class="block w-full border-2 border-gray-400 rounded-md p-3 text-base font-semibold focus:ring-2 focus:ring-teal-500 focus:border-teal-500 bg-white">
                            <?php for ($a = date('Y'); $a >= 2020; $a--): ?>
                            <option value="<?php echo $a; ?>" <?php echo $ano_letivo == $a ? 'selected' : ''; ?>><?php echo $a; ?></option>
                            <?php endfor; ?>
                        </select>
                    </div>
                    
                    <div class="flex items-end">
                        <button type="submit" class="w-full bg-teal-600 text-white px-4 py-3 rounded-md font-semibold hover:bg-teal-700 shadow-md hover:shadow-lg transition-all">
                            <i class="fas fa-search mr-2"></i>Buscar
                        </button>
                    </div>
                </form>
            </div>

            <!-- Modo Individual: Relatório de uma turma -->
            <?php if ($modo_visualizacao === 'individual' && $escola_id && $serie_id && $turma_id): ?>
            <div class="bg-gradient-to-r from-emerald-50 to-teal-50 border-2 border-emerald-400 rounded-lg p-6 mb-6 shadow-md">
                <h3 class="text-xl font-bold text-emerald-800 mb-3">
                    <i class="fas fa-file-download mr-2"></i>4. Gerar Relatório Individual
                </h3>
                <div class="bg-white rounded-lg p-4 mb-4 border-l-4 border-emerald-500">
                    <p class="text-gray-800">
                        <i class="fas fa-school text-emerald-600 mr-2"></i><strong>Turma:</strong> <?php echo htmlspecialchars($escola_nome . ' > ' . $serie_nome . ' > Turma ' . $turma_nome); ?>
                    </p>
                    <p class="text-gray-700 mt-1">
                        <i class="fas fa-calendar-check text-emerald-600 mr-2"></i><strong>Período:</strong> <?php echo $bimestre ? $bimestre . 'º Bimestre' : 'Todos os Bimestres'; ?> | <strong>Ano:</strong> <?php echo $ano_letivo; ?>
                    </p>
                </div>
                <div class="flex gap-4 flex-wrap">
                    <a href="?page=relatorio_visualizar&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&serie=<?php echo $serie_id; ?>&turma=<?php echo $turma_id; ?>&bimestre=<?php echo $bimestre; ?>&ano_letivo=<?php echo $ano_letivo; ?>" 
                       class="bg-teal-600 text-white px-6 py-3 rounded-md font-semibold hover:bg-teal-700 shadow-md hover:shadow-lg transition-all inline-flex items-center">
                        <i class="fas fa-eye mr-2"></i>Visualizar Relatório
                    </a>
                    <a href="?page=relatorio_pdf&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&serie=<?php echo $serie_id; ?>&turma=<?php echo $turma_id; ?>&bimestre=<?php echo $bimestre; ?>&ano_letivo=<?php echo $ano_letivo; ?>" 
                       target="_blank"
                       class="bg-red-600 text-white px-6 py-3 rounded-md font-semibold hover:bg-red-700 shadow-md hover:shadow-lg transition-all inline-flex items-center">
                        <i class="fas fa-file-pdf mr-2"></i>Baixar PDF
                    </a>
                    <a href="?page=relatorio_excel&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&serie=<?php echo $serie_id; ?>&turma=<?php echo $turma_id; ?>&bimestre=<?php echo $bimestre; ?>&ano_letivo=<?php echo $ano_letivo; ?>" 
                       class="bg-green-600 text-white px-6 py-3 rounded-md font-semibold hover:bg-green-700 shadow-md hover:shadow-lg transition-all inline-flex items-center">
                        <i class="fas fa-file-excel mr-2"></i>Exportar Excel
                    </a>
                </div>
            </div>
            <?php endif; ?>

            <!-- Modo Todos: Lista de todas as turmas -->
            <?php if ($modo_visualizacao === 'todos' && $tipo_relatorio && $escola_id && count($dados_relatorios) > 0): ?>
            <div class="bg-white border-2 border-gray-300 rounded-lg overflow-hidden">
                <div class="bg-gradient-to-r from-purple-100 to-indigo-100 px-6 py-4 border-b">
                    <h3 class="text-lg font-bold text-gray-800">
                        <i class="fas fa-list mr-2"></i>Todas as Turmas - <?php echo htmlspecialchars($escola_nome); ?>
                    </h3>
                    <p class="text-sm text-gray-600 mt-1">
                        Período: <?php echo $bimestre ? $bimestre . 'º Bimestre' : 'Todos os Bimestres'; ?> | Ano: <?php echo $ano_letivo; ?> | 
                        Total: <?php echo count($dados_relatorios); ?> registro(s)
                    </p>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50 border-b-2 border-gray-200">
                            <tr>
                                <?php if ($tipo_relatorio !== 'supervisor'): ?>
                                <th class="px-6 py-3 text-left text-xs font-bold text-gray-700 uppercase">Série</th>
                                <?php endif; ?>
                                <th class="px-6 py-3 text-left text-xs font-bold text-gray-700 uppercase">Turma</th>
                                <?php if ($bimestre): ?>
                                <th class="px-6 py-3 text-center text-xs font-bold text-gray-700 uppercase">Bimestre</th>
                                <?php endif; ?>
                                <th class="px-6 py-3 text-center text-xs font-bold text-gray-700 uppercase">Total Alunos</th>
                                <th class="px-6 py-3 text-center text-xs font-bold text-gray-700 uppercase">Necessidades Especiais</th>
                                <th class="px-6 py-3 text-center text-xs font-bold text-gray-700 uppercase">Ações</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php foreach ($dados_relatorios as $dado): ?>
                            <tr class="hover:bg-gray-50">
                                <?php if ($tipo_relatorio !== 'supervisor'): ?>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                    <?php echo htmlspecialchars($dado['nome_serie']); ?>
                                </td>
                                <?php endif; ?>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                    <?php echo htmlspecialchars($dado['nome_turma']); ?>
                                </td>
                                <?php if ($bimestre): ?>
                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-semibold text-purple-600">
                                    <?php echo $dado['bimestre']; ?>º
                                </td>
                                <?php endif; ?>
                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-semibold text-blue-600">
                                    <?php echo $dado['total_alunos'] ?? 0; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-semibold text-orange-600">
                                    <?php echo $dado['alunos_necessidades_especiais'] ?? $dado['necessidades_especiais'] ?? $dado['alunos_com_laudo'] ?? 0; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm">
                                    <div class="flex justify-center gap-2">
                                        <a href="?page=relatorio_visualizar&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&serie=<?php echo $dado['serie_id']; ?>&turma=<?php echo $dado['turma_id']; ?>&bimestre=<?php echo $dado['bimestre'] ?? ''; ?>&ano_letivo=<?php echo $ano_letivo; ?>" 
                                           class="bg-teal-600 text-white px-3 py-1 rounded text-xs font-semibold hover:bg-teal-700" title="Visualizar">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="?page=relatorio_pdf&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&serie=<?php echo $dado['serie_id']; ?>&turma=<?php echo $dado['turma_id']; ?>&bimestre=<?php echo $dado['bimestre'] ?? ''; ?>&ano_letivo=<?php echo $ano_letivo; ?>" 
                                           target="_blank"
                                           class="bg-red-600 text-white px-3 py-1 rounded text-xs font-semibold hover:bg-red-700" title="PDF">
                                            <i class="fas fa-file-pdf"></i>
                                        </a>
                                        <a href="?page=relatorio_excel&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&serie=<?php echo $dado['serie_id']; ?>&turma=<?php echo $dado['turma_id']; ?>&bimestre=<?php echo $dado['bimestre'] ?? ''; ?>&ano_letivo=<?php echo $ano_letivo; ?>" 
                                           class="bg-green-600 text-white px-3 py-1 rounded text-xs font-semibold hover:bg-green-700" title="Excel">
                                            <i class="fas fa-file-excel"></i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Botão para exportar tudo -->
                <div class="bg-gray-50 px-6 py-4 border-t flex justify-end gap-3">
                    <a href="?page=relatorio_excel&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&bimestre=<?php echo $bimestre; ?>&ano_letivo=<?php echo $ano_letivo; ?>&modo=todos" 
                       class="bg-green-600 text-white px-6 py-2 rounded-md font-semibold hover:bg-green-700 inline-flex items-center">
                        <i class="fas fa-download mr-2"></i>Exportar Tudo (Excel)
                    </a>
                </div>
            </div>
            <?php elseif ($modo_visualizacao === 'todos' && $tipo_relatorio && $escola_id): ?>
            <div class="text-center py-12 bg-yellow-50 border-2 border-yellow-200 rounded-lg">
                <i class="fas fa-inbox text-6xl text-yellow-300 mb-4"></i>
                <p class="text-gray-600 text-lg">Nenhum dado encontrado para esta seleção</p>
            </div>
            <?php endif; ?>

            <!-- Modo Geral: Estatísticas por série -->
            <?php if ($modo_visualizacao === 'geral' && $tipo_relatorio && $escola_id && count($dados_geral) > 0): ?>
            <div class="bg-white border-2 border-gray-300 rounded-lg overflow-hidden">
                <div class="bg-gradient-to-r from-pink-100 to-rose-100 px-6 py-4 border-b">
                    <h3 class="text-lg font-bold text-gray-800">
                        <i class="fas fa-chart-pie mr-2"></i>Relatório Geral - Estatísticas por Série
                    </h3>
                    <p class="text-sm text-gray-600 mt-1">
                        Escola: <?php echo htmlspecialchars($escola_nome); ?> | 
                        Período: <?php echo $bimestre ? $bimestre . 'º Bimestre' : 'Todos os Bimestres'; ?> | 
                        Ano: <?php echo $ano_letivo; ?>
                    </p>
                </div>
                
                <div class="p-6">
                    <!-- Cards de Estatísticas -->
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                        <?php
                        $total_geral_turmas = 0;
                        $total_geral_alunos = 0;
                        $total_geral_necessidades = 0;
                        foreach ($dados_geral as $dado) {
                            $total_geral_turmas += $dado['total_turmas'];
                            $total_geral_alunos += $dado['total_alunos'];
                            $total_geral_necessidades += $dado['total_necessidades'];
                        }
                        ?>
                        <div class="bg-gradient-to-br from-blue-50 to-blue-100 border-2 border-blue-300 rounded-lg p-6 text-center">
                            <div class="text-4xl font-bold text-blue-600 mb-2"><?php echo $total_geral_turmas; ?></div>
                            <div class="text-sm font-semibold text-gray-700">Total de Turmas</div>
                        </div>
                        <div class="bg-gradient-to-br from-green-50 to-green-100 border-2 border-green-300 rounded-lg p-6 text-center">
                            <div class="text-4xl font-bold text-green-600 mb-2"><?php echo $total_geral_alunos; ?></div>
                            <div class="text-sm font-semibold text-gray-700">Total de Alunos</div>
                        </div>
                        <div class="bg-gradient-to-br from-orange-50 to-orange-100 border-2 border-orange-300 rounded-lg p-6 text-center">
                            <div class="text-4xl font-bold text-orange-600 mb-2"><?php echo $total_geral_necessidades; ?></div>
                            <div class="text-sm font-semibold text-gray-700">Necessidades Especiais</div>
                        </div>
                    </div>
                    
                    <!-- Tabela por Série -->
                    <div class="overflow-x-auto">
                        <table class="w-full border rounded-lg overflow-hidden">
                            <thead class="bg-gradient-to-r from-pink-100 to-rose-100 border-b-2 border-gray-300">
                                <tr>
                                    <th class="px-6 py-3 text-left text-sm font-bold text-gray-800">
                                        <?php echo $tipo_relatorio === 'supervisor' ? 'Turma/Período' : 'Série'; ?>
                                    </th>
                                    <th class="px-6 py-3 text-center text-sm font-bold text-gray-800">
                                        <?php echo $tipo_relatorio === 'supervisor' ? 'Registros' : 'Turmas'; ?>
                                    </th>
                                    <th class="px-6 py-3 text-center text-sm font-bold text-gray-800">Total Alunos</th>
                                    <th class="px-6 py-3 text-center text-sm font-bold text-gray-800">
                                        <?php echo $tipo_relatorio === 'supervisor' ? 'Com Laudo' : 'Necessidades Especiais'; ?>
                                    </th>
                                    <th class="px-6 py-3 text-center text-sm font-bold text-gray-800">
                                        <?php echo $tipo_relatorio === 'supervisor' ? '% Com Laudo' : '% Necessidades'; ?>
                                    </th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php foreach ($dados_geral as $dado): ?>
                                <?php 
                                $percentual = $dado['total_alunos'] > 0 
                                    ? round(($dado['total_necessidades'] / $dado['total_alunos']) * 100, 1) 
                                    : 0;
                                ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 text-sm font-semibold text-gray-900">
                                        <?php echo htmlspecialchars($dado['nome_serie']); ?>
                                    </td>
                                    <td class="px-6 py-4 text-center text-sm font-semibold text-blue-600">
                                        <?php echo $dado['total_turmas']; ?>
                                    </td>
                                    <td class="px-6 py-4 text-center text-sm font-semibold text-green-600">
                                        <?php echo $dado['total_alunos']; ?>
                                    </td>
                                    <td class="px-6 py-4 text-center text-sm font-semibold text-orange-600">
                                        <?php echo $dado['total_necessidades']; ?>
                                    </td>
                                    <td class="px-6 py-4 text-center text-sm font-semibold text-purple-600">
                                        <?php echo $percentual; ?>%
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="bg-gray-100 font-bold">
                                    <td class="px-6 py-4 text-sm text-gray-900">TOTAL GERAL</td>
                                    <td class="px-6 py-4 text-center text-sm text-blue-700"><?php echo $total_geral_turmas; ?></td>
                                    <td class="px-6 py-4 text-center text-sm text-green-700"><?php echo $total_geral_alunos; ?></td>
                                    <td class="px-6 py-4 text-center text-sm text-orange-700"><?php echo $total_geral_necessidades; ?></td>
                                    <td class="px-6 py-4 text-center text-sm text-purple-700">
                                        <?php echo $total_geral_alunos > 0 ? round(($total_geral_necessidades / $total_geral_alunos) * 100, 1) : 0; ?>%
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <!-- Botão para exportar estatísticas -->
                <div class="bg-gray-50 px-6 py-4 border-t flex justify-end gap-3">
                    <a href="?page=relatorio_pdf&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&bimestre=<?php echo $bimestre; ?>&ano_letivo=<?php echo $ano_letivo; ?>&modo=geral" 
                       target="_blank"
                       class="bg-red-600 text-white px-6 py-2 rounded-md font-semibold hover:bg-red-700 inline-flex items-center">
                        <i class="fas fa-file-pdf mr-2"></i>Exportar Relatório Geral (PDF)
                    </a>
                    <a href="?page=relatorio_excel&tipo=<?php echo $tipo_relatorio; ?>&escola=<?php echo $escola_id; ?>&bimestre=<?php echo $bimestre; ?>&ano_letivo=<?php echo $ano_letivo; ?>&modo=geral" 
                       class="bg-green-600 text-white px-6 py-2 rounded-md font-semibold hover:bg-green-700 inline-flex items-center">
                        <i class="fas fa-file-excel mr-2"></i>Exportar Relatório Geral (Excel)
                    </a>
                </div>
            </div>
            <?php elseif ($modo_visualizacao === 'geral' && $tipo_relatorio && $escola_id): ?>
            <div class="text-center py-12 bg-yellow-50 border-2 border-yellow-200 rounded-lg">
                <i class="fas fa-inbox text-6xl text-yellow-300 mb-4"></i>
                <p class="text-gray-600 text-lg">Nenhum dado encontrado para esta seleção</p>
            </div>
            <?php endif; ?>

            <!-- Mensagem inicial -->
            <?php if (!$tipo_relatorio): ?>
            <div class="text-center py-16 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg border-2 border-blue-200">
                <div class="max-w-md mx-auto">
                    <i class="fas fa-hand-pointer text-6xl text-blue-400 mb-4 animate-pulse"></i>
                    <h3 class="text-2xl font-bold text-gray-800 mb-3">Bem-vindo ao Sistema de Relatórios!</h3>
                    <p class="text-gray-600 text-lg mb-4">Para começar, siga os passos acima:</p>
                    <div class="text-left bg-white rounded-lg p-5 shadow-sm">
                        <div class="flex items-start mb-3">
                            <div class="bg-teal-500 text-white rounded-full w-8 h-8 flex items-center justify-center font-bold mr-3 flex-shrink-0">1</div>
                            <div>
                                <p class="font-semibold text-gray-800">Selecione o Tipo de Condensado</p>
                                <p class="text-sm text-gray-600">Escolha qual condensado deseja gerar</p>
                            </div>
                        </div>
                        <div class="flex items-start mb-3">
                            <div class="bg-indigo-500 text-white rounded-full w-8 h-8 flex items-center justify-center font-bold mr-3 flex-shrink-0">2</div>
                            <div>
                                <p class="font-semibold text-gray-800">Escolha o Modo de Visualização</p>
                                <p class="text-sm text-gray-600">Individual, lista ou estatísticas</p>
                            </div>
                        </div>
                        <div class="flex items-start">
                            <div class="bg-gray-500 text-white rounded-full w-8 h-8 flex items-center justify-center font-bold mr-3 flex-shrink-0">3</div>
                            <div>
                                <p class="font-semibold text-gray-800">Defina os Filtros</p>
                                <p class="text-sm text-gray-600">Configure escola, turma, bimestre e ano</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <?php endif; ?>
        </div>
    </div>
</div>