<?php
// Seed script to insert provided Schools, Series, and Turmas

// Carrega configuração centralizada
require_once __DIR__ . '/../config/database.php';

$dsn = getDBDSN();
$user = DB_USER;
$pass = DB_PASS;
$options = DB_OPTIONS;

try {
    $pdo = new PDO($dsn, $user, $pass, $options);
} catch (PDOException $e) {
    die('DB connection error: ' . $e->getMessage());
}

// Ensure tables exist
$pdo->exec("CREATE TABLE IF NOT EXISTS escolas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uniq_nome (nome)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

$pdo->exec("CREATE TABLE IF NOT EXISTS escola_series (
    id INT AUTO_INCREMENT PRIMARY KEY,
    escola_id INT NOT NULL,
    nome_serie VARCHAR(50) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uniq_escola_serie (escola_id, nome_serie),
    KEY idx_escola (escola_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

$pdo->exec("CREATE TABLE IF NOT EXISTS turmas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    escola_id INT NOT NULL,
    serie_id INT NOT NULL,
    nome_turma VARCHAR(50) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uniq_turma (escola_id, serie_id, nome_turma),
    KEY idx_escola (escola_id),
    KEY idx_serie (serie_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

// Provided data
$data = array(
    'CEMEI TIA FIA' => array(
        'MATERNAL -1',
        'MATERNAL-2',
        'MATERNAL-3',
        'MATERNAL -4',
        '1º PERÍODO-1',
        '1ºPERÍODO-2',
        '1º PERÍODO-3',
        '2º PERÍODO-1',
        '2º PERÍODO-2',
        '2º PERÍODO-3',
    ),
    'CEMEI RICARDO ROCHA VIANA' => array(
        'MATERNAL -1',
        'MATERNAL-2',
        'MATERNAL-3',
        '1º PERÍODO-1',
        '1ºPERÍODO-2',
        '1º PERÍODO-3',
        '2º PERÍODO-1',
        '2º PERÍODO-2',
    ),
    'CEMEI MARIA SILVESTRE DE SOUZA' => array(
        'MATERNAL- 1',
        'MATERNAL -2',
        '1º PERÍODO -1',
        '2º PERÍODO-1',
        '2ª PERÍODO-2',
    ),
    'CEMEI ANA DE LOURDES NUNES' => array(
        'MATERNAL- 1',
        '1º PERÍODO -1',
        '2º PERÍODO-1',
    ),
    'CEMEI CRISTOVÃO RODRIGUES CARLOS' => array(
        'MATERNAL- 1',
        '1º PERÍODO -1',
        '2º PERÍODO-1',
    ),
);

function normalize_entry($entry) {
    // Collapse spaces
    $entry = preg_replace('/\s+/', ' ', trim($entry));
    // Normalize spaces around hyphen
    $entry = preg_replace('/\s*-\s*/', '-', $entry);
    // Ensure space before PERÍODO if missing (e.g., 1ºPERÍODO -> 1º PERÍODO)
    $entry = preg_replace('/(º)(PERÍODO)/u', '$1 $2', $entry);
    $entry = preg_replace('/(ª)(PERÍODO)/u', '$1 $2', $entry);
    return $entry;
}

function split_serie_turma($entry) {
    // Expect pattern "SERIE-TURMA" where TURMA is typically a small token (e.g., 1, 2, A)
    $parts = explode('-', $entry);
    if (count($parts) >= 2) {
        $turma = array_pop($parts);
        $serie = implode('-', $parts);
        return array(trim($serie), trim($turma));
    }
    // Fallback: everything is série, turma empty
    return array(trim($entry), '');
}

function get_or_create_escola_id(PDO $pdo, $nome) {
    $st = $pdo->prepare('SELECT id FROM escolas WHERE nome = ?');
    $st->execute(array($nome));
    $row = $st->fetch();
    if ($row) return (int)$row['id'];
    $ins = $pdo->prepare('INSERT INTO escolas (nome) VALUES (?)');
    $ins->execute(array($nome));
    return (int)$pdo->lastInsertId();
}

function get_or_create_serie_id(PDO $pdo, $escola_id, $nome_serie) {
    $st = $pdo->prepare('SELECT id FROM escola_series WHERE escola_id = ? AND nome_serie = ?');
    $st->execute(array($escola_id, $nome_serie));
    $row = $st->fetch();
    if ($row) return (int)$row['id'];
    $ins = $pdo->prepare('INSERT INTO escola_series (escola_id, nome_serie) VALUES (?, ?)');
    $ins->execute(array($escola_id, $nome_serie));
    return (int)$pdo->lastInsertId();
}

function ensure_turma(PDO $pdo, $escola_id, $serie_id, $nome_turma) {
    if ($nome_turma === '') return;
    $st = $pdo->prepare('SELECT id FROM turmas WHERE escola_id = ? AND serie_id = ? AND nome_turma = ?');
    $st->execute(array($escola_id, $serie_id, $nome_turma));
    if ($st->fetch()) return;
    $ins = $pdo->prepare('INSERT INTO turmas (escola_id, serie_id, nome_turma) VALUES (?, ?, ?)');
    $ins->execute(array($escola_id, $serie_id, $nome_turma));
}

$created_escolas = 0; $created_series = 0; $created_turmas = 0;

foreach ($data as $escola_nome => $entries) {
    $eid_before = $pdo->prepare('SELECT id FROM escolas WHERE nome = ?');
    $eid_before->execute(array($escola_nome));
    $had_escola = (bool)$eid_before->fetch();

    $escola_id = get_or_create_escola_id($pdo, $escola_nome);
    if (!$had_escola) $created_escolas++;

    foreach ($entries as $raw) {
        $norm = normalize_entry($raw);
        list($serie, $turma) = split_serie_turma($norm);
        // Normalize series spacing further: ensure single spaces
        $serie = preg_replace('/\s+/', ' ', trim($serie));
        $turma = preg_replace('/\s+/', ' ', trim($turma));

        // Unify known variants: "1ºPERÍODO" -> "1º PERÍODO"
        $serie = preg_replace('/(\d+º)\s*(PERÍODO)/u', '$1 $2', $serie);

        // Create or fetch série
        $sid_before = $pdo->prepare('SELECT id FROM escola_series WHERE escola_id = ? AND nome_serie = ?');
        $sid_before->execute(array($escola_id, $serie));
        $had_serie = (bool)$sid_before->fetch();
        $serie_id = get_or_create_serie_id($pdo, $escola_id, $serie);
        if (!$had_serie) $created_series++;

        // Create turma
        $tid_before = $pdo->prepare('SELECT id FROM turmas WHERE escola_id = ? AND serie_id = ? AND nome_turma = ?');
        $tid_before->execute(array($escola_id, $serie_id, $turma));
        $had_turma = (bool)$tid_before->fetch();
        ensure_turma($pdo, $escola_id, $serie_id, $turma);
        if (!$had_turma && $turma !== '') $created_turmas++;

        echo "[$escola_nome] Série: $serie | Turma: $turma -> OK\n";
    }
}

echo "\nResumo: escolas criadas=$created_escolas, séries criadas=$created_series, turmas criadas=$created_turmas\n";
