<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Usuarios extends CI_Controller {

    public function __construct()
    {
        parent::__construct();
        $this->load->library('Auth_lib');
        $this->load->model('Usuario_model');
        
        // Verifica se está logado
        $this->auth_lib->verificar_login();
        
        // Verifica permissão de administrador (nível 9)
        $this->auth_lib->verificar_permissao(9);
    }

    /**
     * Lista todos os usuários
     */
    public function index()
    {
        $data = array(
            'titulo' => 'Gerenciar Usuários - Painel Administrativo SIMAE',
            'usuario_logado' => $this->auth_lib->usuario(),
            'usuarios' => $this->Usuario_model->listar_usuarios(),
            'sucesso' => $this->session->flashdata('sucesso'),
            'erro' => $this->session->flashdata('erro')
        );

        $this->load->view('admin/layout/header', $data);
        $this->load->view('admin/usuarios/index', $data);
        $this->load->view('admin/layout/footer');
    }

    /**
     * Formulário para novo usuário
     */
    public function novo()
    {
        if ($this->input->post()) {
            $this->processar_novo_usuario();
        }

        $data = array(
            'titulo' => 'Novo Usuário - Painel Administrativo SIMAE',
            'usuario_logado' => $this->auth_lib->usuario(),
            'erro' => $this->session->flashdata('erro')
        );

        $this->load->view('admin/layout/header', $data);
        $this->load->view('admin/usuarios/form', $data);
        $this->load->view('admin/layout/footer');
    }

    /**
     * Formulário para editar usuário
     */
    public function editar($id)
    {
        $usuario = $this->Usuario_model->buscar_por_id($id);
        
        if (!$usuario) {
            show_404();
        }

        if ($this->input->post()) {
            $this->processar_editar_usuario($id);
        }

        $data = array(
            'titulo' => 'Editar Usuário - Painel Administrativo SIMAE',
            'usuario_logado' => $this->auth_lib->usuario(),
            'usuario' => $usuario,
            'erro' => $this->session->flashdata('erro')
        );

        $this->load->view('admin/layout/header', $data);
        $this->load->view('admin/usuarios/form', $data);
        $this->load->view('admin/layout/footer');
    }

    /**
     * Remove usuário
     */
    public function remover($id)
    {
        if ($this->Usuario_model->remover_usuario($id)) {
            $this->session->set_flashdata('sucesso', 'Usuário removido com sucesso!');
        } else {
            $this->session->set_flashdata('erro', 'Erro ao remover usuário.');
        }
        
        redirect('admin/usuarios');
    }

    /**
     * Processa formulário de novo usuário
     */
    private function processar_novo_usuario()
    {
        $this->form_validation->set_rules('nome', 'Nome', 'required');
        $this->form_validation->set_rules('email', 'E-mail', 'required|valid_email');
        $this->form_validation->set_rules('senha', 'Senha', 'required|min_length[6]');
        $this->form_validation->set_rules('nivel', 'Nível', 'required|in_list[1,5,9]');

        if ($this->form_validation->run() == FALSE) {
            $this->session->set_flashdata('erro', validation_errors());
            return;
        }

        // Verifica se email já existe
        if ($this->Usuario_model->email_existe($this->input->post('email'))) {
            $this->session->set_flashdata('erro', 'Este e-mail já está em uso.');
            return;
        }

        $dados = array(
            'nome' => $this->input->post('nome'),
            'email' => $this->input->post('email'),
            'senha' => $this->input->post('senha'),
            'nivel' => $this->input->post('nivel'),
            'ativo' => 1
        );

        if ($this->Usuario_model->inserir_usuario($dados)) {
            $this->session->set_flashdata('sucesso', 'Usuário criado com sucesso!');
            redirect('admin/usuarios');
        } else {
            $this->session->set_flashdata('erro', 'Erro ao criar usuário.');
        }
    }

    /**
     * Processa formulário de editar usuário
     */
    private function processar_editar_usuario($id)
    {
        $this->form_validation->set_rules('nome', 'Nome', 'required');
        $this->form_validation->set_rules('email', 'E-mail', 'required|valid_email');
        $this->form_validation->set_rules('nivel', 'Nível', 'required|in_list[1,5,9]');
        
        if (!empty($this->input->post('senha'))) {
            $this->form_validation->set_rules('senha', 'Senha', 'min_length[6]');
        }

        if ($this->form_validation->run() == FALSE) {
            $this->session->set_flashdata('erro', validation_errors());
            return;
        }

        // Verifica se email já existe (excluindo o próprio usuário)
        if ($this->Usuario_model->email_existe($this->input->post('email'), $id)) {
            $this->session->set_flashdata('erro', 'Este e-mail já está em uso.');
            return;
        }

        $dados = array(
            'nome' => $this->input->post('nome'),
            'email' => $this->input->post('email'),
            'nivel' => $this->input->post('nivel'),
            'ativo' => $this->input->post('ativo') ? 1 : 0
        );

        if (!empty($this->input->post('senha'))) {
            $dados['senha'] = $this->input->post('senha');
        }

        if ($this->Usuario_model->atualizar_usuario($id, $dados)) {
            $this->session->set_flashdata('sucesso', 'Usuário atualizado com sucesso!');
            redirect('admin/usuarios');
        } else {
            $this->session->set_flashdata('erro', 'Erro ao atualizar usuário.');
        }
    }
}